<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\StockItemResource;
use App\Models\StockItem;
use App\Services\StockService;
use Illuminate\Http\Request;

class StockItemController extends Controller
{
    public function __construct(private StockService $stockService)
    {
    }

    public function index()
    {
        $items = StockItem::all();
        return StockItemResource::collection($items);
    }

    public function store(Request $request)
    {
        $request->validate([
            'item_name' => 'required|string|max:255',
            'vendor' => 'nullable|string|max:255',
            'stock_in_date' => 'required|date',
            'stock_in_qty' => 'required|integer|min:0',
            'cost_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
        ]);

        $stockItem = StockItem::create([
            'item_name' => $request->item_name,
            'vendor' => $request->vendor,
            'stock_in_date' => $request->stock_in_date,
            'stock_in_qty' => $request->stock_in_qty,
            'stock_balance' => $request->stock_in_qty,
            'cost_price' => $request->cost_price,
            'selling_price' => $request->selling_price,
            'total_stock_rate' => $request->stock_in_qty * $request->cost_price,
        ]);

        return new StockItemResource($stockItem);
    }

    public function show(StockItem $stockItem)
    {
        return new StockItemResource($stockItem);
    }

    public function update(Request $request, StockItem $stockItem)
    {
        $request->validate([
            'item_name' => 'required|string|max:255',
            'vendor' => 'nullable|string|max:255',
            'cost_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
        ]);

        $stockItem->update($request->only(['item_name', 'vendor', 'cost_price', 'selling_price']));
        $stockItem->total_stock_rate = $stockItem->stock_balance * $stockItem->cost_price;
        $stockItem->save();

        return new StockItemResource($stockItem);
    }

    public function destroy(StockItem $stockItem)
    {
        $stockItem->delete();
        return response()->json(['message' => 'Stock item deleted successfully']);
    }

    public function stockIn(Request $request, StockItem $stockItem)
    {
        $request->validate([
            'qty' => 'required|integer|min:1',
            'date' => 'required|date',
        ]);

        $updatedItem = $this->stockService->stockIn($stockItem, $request->qty, $request->date);

        return new StockItemResource($updatedItem);
    }

    public function stockOut(Request $request, StockItem $stockItem)
    {
        $request->validate([
            'qty' => 'required|integer|min:1',
            'date' => 'required|date',
        ]);

        $updatedItem = $this->stockService->stockOut($stockItem, $request->qty, $request->date);

        return new StockItemResource($updatedItem);
    }

    public function alerts()
    {
        $alerts = $this->stockService->getStockAlerts();

        return response()->json([
            'out_of_stock' => StockItemResource::collection($alerts['out_of_stock']),
            'low_stock' => StockItemResource::collection($alerts['low_stock']),
        ]);
    }
}
