<?php

namespace App\Services;

use App\Models\StockItem;
use App\Exceptions\InsufficientStockException;

class StockService
{
    /**
     * Stock in operation - add stock quantity
     */
    public function stockIn(StockItem $stockItem, int $qty, string $date): StockItem
    {
        $stockItem->stock_in_qty += $qty;
        $stockItem->stock_balance += $qty;
        $stockItem->stock_in_date = $date;
        $stockItem->total_stock_rate = $stockItem->stock_balance * $stockItem->cost_price;
        $stockItem->save();

        return $stockItem->fresh();
    }

    /**
     * Stock out operation - remove stock quantity
     */
    public function stockOut(StockItem $stockItem, int $qty, string $date): StockItem
    {
        if ($qty > $stockItem->stock_balance) {
            throw new InsufficientStockException(
                "Insufficient stock for {$stockItem->item_name}. Available: {$stockItem->stock_balance}, Requested: {$qty}"
            );
        }

        $stockItem->stock_out_qty += $qty;
        $stockItem->stock_balance -= $qty;
        $stockItem->stock_out_date = $date;
        $stockItem->total_sales_rate += ($qty * $stockItem->selling_price);
        $stockItem->total_stock_rate = $stockItem->stock_balance * $stockItem->cost_price;
        $stockItem->save();

        return $stockItem->fresh();
    }

    /**
     * Get stock alerts
     */
    public function getStockAlerts(int $lowStockThreshold = 10): array
    {
        return [
            'out_of_stock' => StockItem::outOfStock()->get(),
            'low_stock' => StockItem::lowStock($lowStockThreshold)->get(),
        ];
    }
}
