import { useState } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { StockItem } from "@/data/stockData";
import { ArrowDownToLine, ArrowUpFromLine } from "lucide-react";

interface StockInOutDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  item: StockItem | null;
  mode: "in" | "out";
  onConfirm: (qty: number, date: string) => void;
}

const StockInOutDialog = ({ open, onOpenChange, item, mode, onConfirm }: StockInOutDialogProps) => {
  const [qty, setQty] = useState("");
  const [date, setDate] = useState(new Date().toLocaleDateString("en-GB"));

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    const quantity = Number(qty);
    if (!quantity || quantity <= 0) return;
    if (mode === "out" && item && quantity > item.stockBalance) return;
    onConfirm(quantity, date);
    setQty("");
    setDate(new Date().toLocaleDateString("en-GB"));
    onOpenChange(false);
  };

  const isIn = mode === "in";

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-sm">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            {isIn ? (
              <ArrowDownToLine className="h-5 w-5 text-success" />
            ) : (
              <ArrowUpFromLine className="h-5 w-5 text-warning" />
            )}
            Stock {isIn ? "In" : "Out"}: {item?.itemName}
          </DialogTitle>
          <DialogDescription>
            {isIn
              ? "Add new stock quantity to inventory."
              : `Remove stock from inventory. Available: ${item?.stockBalance ?? 0} units.`}
          </DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="stockQty">Quantity</Label>
            <Input
              id="stockQty"
              type="number"
              min="1"
              max={mode === "out" ? item?.stockBalance : undefined}
              value={qty}
              onChange={(e) => setQty(e.target.value)}
              placeholder="Enter quantity"
              required
            />
            {mode === "out" && item && Number(qty) > item.stockBalance && (
              <p className="text-xs text-destructive">Cannot exceed available balance ({item.stockBalance})</p>
            )}
          </div>
          <div className="space-y-2">
            <Label htmlFor="stockDate">Date</Label>
            <Input
              id="stockDate"
              value={date}
              onChange={(e) => setDate(e.target.value)}
              placeholder="DD/MM/YYYY"
            />
          </div>
          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button
              type="submit"
              className={isIn ? "bg-success hover:bg-success/90 text-success-foreground" : ""}
            >
              {isIn ? "Stock In" : "Stock Out"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  );
};

export default StockInOutDialog;
