import { useState } from "react";
import { StockItem } from "@/data/stockData";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import {
  Plus,
  Pencil,
  Trash2,
  ArrowDownToLine,
  ArrowUpFromLine,
  Search,
} from "lucide-react";
import { Input } from "@/components/ui/input";
import { StockManagerActions } from "@/hooks/useStockManager";
import AddStockDialog from "./AddStockDialog";
import EditStockDialog from "./EditStockDialog";
import DeleteStockDialog from "./DeleteStockDialog";
import StockInOutDialog from "./StockInOutDialog";
import { useToast } from "@/hooks/use-toast";

interface StockTableProps {
  items: StockItem[];
  actions: StockManagerActions;
}

const StockTable = ({ items = [], actions }: StockTableProps) => {
  const { toast } = useToast();
  const [search, setSearch] = useState("");
  const [addOpen, setAddOpen] = useState(false);
  const [editOpen, setEditOpen] = useState(false);
  const [deleteOpen, setDeleteOpen] = useState(false);
  const [stockInOutOpen, setStockInOutOpen] = useState(false);
  const [stockMode, setStockMode] = useState<"in" | "out">("in");
  const [selectedIndex, setSelectedIndex] = useState<number | null>(null);

  const selectedItem = selectedIndex !== null ? items[selectedIndex] : null;

  const filtered = items.filter((item) =>
    item.itemName.toLowerCase().includes(search.toLowerCase())
  );

  const getStockStatus = (balance: number, initial: number) => {
    if (balance === 0) return { label: "Out of Stock", variant: "destructive" as const };
    if (balance / initial < 0.2) return { label: "Low Stock", variant: "secondary" as const };
    return { label: "In Stock", variant: "default" as const };
  };

  const handleEdit = (globalIndex: number) => {
    setSelectedIndex(globalIndex);
    setEditOpen(true);
  };

  const handleDelete = (globalIndex: number) => {
    setSelectedIndex(globalIndex);
    setDeleteOpen(true);
  };

  const handleStockInOut = (globalIndex: number, mode: "in" | "out") => {
    setSelectedIndex(globalIndex);
    setStockMode(mode);
    setStockInOutOpen(true);
  };

  // Get the real index in the full items array
  const getRealIndex = (filteredItem: StockItem) => {
    return items.findIndex((i) => i === filteredItem);
  };

  return (
    <>
      <div className="glass-card rounded-xl overflow-hidden animate-fade-in" style={{ animationDelay: "200ms" }}>
        {/* Header */}
        <div className="p-5 border-b border-border flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
          <div>
            <h3 className="text-lg font-bold font-display text-foreground">Stock Balance</h3>
            <p className="text-sm text-muted-foreground mt-1">
              {items.length} items · {items.filter((i) => i.stockBalance === 0).length} out of stock
            </p>
          </div>
          <div className="flex items-center gap-2">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                placeholder="Search items..."
                className="pl-9 w-48"
              />
            </div>
            <Button onClick={() => setAddOpen(true)} size="sm" className="gap-1.5">
              <Plus className="h-4 w-4" />
              <span className="hidden sm:inline">Add Item</span>
            </Button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow className="bg-muted/50">
                <TableHead className="font-semibold text-foreground">Item Name</TableHead>
                <TableHead className="font-semibold text-foreground">Vendor</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Stock In</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Sold</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Balance</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Cost (৳)</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Price (৳)</TableHead>
                <TableHead className="text-right font-semibold text-foreground">Sales (৳)</TableHead>
                <TableHead className="text-center font-semibold text-foreground">Status</TableHead>
                <TableHead className="text-center font-semibold text-foreground">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filtered.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={10} className="text-center py-8 text-muted-foreground">
                    {search ? "No items match your search." : "No stock items yet. Add your first item!"}
                  </TableCell>
                </TableRow>
              ) : (
                filtered.map((item) => {
                  const realIndex = getRealIndex(item);
                  const status = getStockStatus(item.stockBalance, item.stockInQty);
                  return (
                    <TableRow key={realIndex} className="hover:bg-muted/30 transition-colors">
                      <TableCell className="font-medium text-foreground">{item.itemName}</TableCell>
                      <TableCell className="text-muted-foreground">{item.vendor || "—"}</TableCell>
                      <TableCell className="text-right text-muted-foreground">{item.stockInQty}</TableCell>
                      <TableCell className="text-right text-muted-foreground">{item.stockOutQty}</TableCell>
                      <TableCell className="text-right font-semibold text-foreground">{item.stockBalance}</TableCell>
                      <TableCell className="text-right text-muted-foreground">৳{item.costPrice}</TableCell>
                      <TableCell className="text-right text-muted-foreground">৳{item.sellingPrice}</TableCell>
                      <TableCell className="text-right font-semibold text-primary">
                        ৳{item.totalSalesRate.toLocaleString()}
                      </TableCell>
                      <TableCell className="text-center">
                        <Badge
                          variant={status.variant}
                          className={
                            status.variant === "destructive"
                              ? "bg-destructive/10 text-destructive border-destructive/20"
                              : status.variant === "secondary"
                              ? "bg-warning/10 text-warning border-warning/20"
                              : "bg-success/10 text-success border-success/20"
                          }
                        >
                          {status.label}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center justify-center gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-8 w-8 text-success hover:text-success hover:bg-success/10"
                            title="Stock In"
                            onClick={() => handleStockInOut(realIndex, "in")}
                          >
                            <ArrowDownToLine className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-8 w-8 text-warning hover:text-warning hover:bg-warning/10"
                            title="Stock Out"
                            onClick={() => handleStockInOut(realIndex, "out")}
                          >
                            <ArrowUpFromLine className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-8 w-8 text-primary hover:text-primary hover:bg-primary/10"
                            title="Edit"
                            onClick={() => handleEdit(realIndex)}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-8 w-8 text-destructive hover:text-destructive hover:bg-destructive/10"
                            title="Delete"
                            onClick={() => handleDelete(realIndex)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  );
                })
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      {/* Dialogs */}
      <AddStockDialog
        open={addOpen}
        onOpenChange={setAddOpen}
        onAdd={(item) => {
          actions.addItem(item);
          toast({ title: "Item Added", description: `${item.itemName} has been added to inventory.` });
        }}
      />

      <EditStockDialog
        open={editOpen}
        onOpenChange={setEditOpen}
        item={selectedItem}
        onSave={(item) => {
          if (selectedIndex !== null) {
            actions.updateItem(selectedIndex, item);
            toast({ title: "Item Updated", description: `${item.itemName} has been updated.` });
          }
        }}
      />

      <DeleteStockDialog
        open={deleteOpen}
        onOpenChange={setDeleteOpen}
        item={selectedItem}
        onConfirm={() => {
          if (selectedIndex !== null) {
            const name = items[selectedIndex]?.itemName;
            actions.deleteItem(selectedIndex);
            toast({ title: "Item Deleted", description: `${name} has been removed.`, variant: "destructive" });
          }
          setDeleteOpen(false);
        }}
      />

      <StockInOutDialog
        open={stockInOutOpen}
        onOpenChange={setStockInOutOpen}
        item={selectedItem}
        mode={stockMode}
        onConfirm={(qty, date) => {
          if (selectedIndex !== null) {
            if (stockMode === "in") {
              actions.stockIn(selectedIndex, qty, date);
              toast({ title: "Stock In", description: `Added ${qty} units to ${selectedItem?.itemName}.` });
            } else {
              actions.stockOut(selectedIndex, qty, date);
              toast({ title: "Stock Out", description: `Removed ${qty} units from ${selectedItem?.itemName}.` });
            }
          }
        }}
      />
    </>
  );
};

export default StockTable;
