import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { stockApi, StockItem } from '@/lib/api';
import { toast } from 'sonner';
import { useMemo } from 'react';

export const useStockItems = () => {
  const queryClient = useQueryClient();

  const { data, isLoading, error } = useQuery({
    queryKey: ['stock-items'],
    queryFn: async () => {
      const response = await stockApi.getAll();
      return response.data.data;
    },
  });

  const addItem = useMutation({
    mutationFn: (item: any) => stockApi.create(item),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Stock item added successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to add stock item');
    },
  });

  const updateItem = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      stockApi.update(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Stock item updated successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to update stock item');
    },
  });

  const deleteItem = useMutation({
    mutationFn: (id: number) => stockApi.delete(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Stock item deleted successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to delete stock item');
    },
  });

  const stockIn = useMutation({
    mutationFn: ({ id, qty, date }: { id: number; qty: number; date: string }) =>
      stockApi.stockIn(id, qty, date),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Stock added successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to add stock');
    },
  });

  const stockOut = useMutation({
    mutationFn: ({ id, qty, date }: { id: number; qty: number; date: string }) =>
      stockApi.stockOut(id, qty, date),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Stock removed successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to remove stock');
    },
  });

  // Calculate stats from data
  const stats = useMemo(() => {
    if (!data) {
      return {
        totalProducts: 0,
        totalBalance: 0,
        stockValue: 0,
        salesValue: 0,
        outOfStock: 0,
        lowStock: 0,
      };
    }

    return {
      totalProducts: data.length,
      totalBalance: data.reduce((sum: number, item: StockItem) => sum + item.stockBalance, 0),
      stockValue: data.reduce((sum: number, item: StockItem) => sum + item.totalStockRate, 0),
      salesValue: data.reduce((sum: number, item: StockItem) => sum + item.totalSalesRate, 0),
      outOfStock: data.filter((item: StockItem) => item.stockBalance === 0).length,
      lowStock: data.filter((item: StockItem) => item.stockBalance > 0 && item.stockBalance <= 10).length,
    };
  }, [data]);

  return {
    items: data || [],
    stats,
    isLoading,
    error,
    actions: {
      addItem: (item: any) => addItem.mutate(item),
      updateItem: (id: number, item: any) => updateItem.mutate({ id, data: item }),
      deleteItem: (id: number) => deleteItem.mutate(id),
      stockIn: (id: number, qty: number, date: string) => stockIn.mutate({ id, qty, date }),
      stockOut: (id: number, qty: number, date: string) => stockOut.mutate({ id, qty, date }),
    },
  };
};
